
function applyRewriteSettings(src, target)
	if not src then
		return
	end
	
    for i, v in pairs(src) do
        if type(v) == 'table' then
            target[i] = {}
            copyTable(v, target[i])
        else
            target[i] = v
        end
    end
end

function is_oct( c )
	return c >= 0x30 and c <= 0x37 -- 1-7
end


function custom_input_filter_octal( text ) -- filter oct 1-7 from utf8 input 
	local ret = ""
	for i = 1, #text do
		local c = string.byte(text,i)
		if is_oct(c)  then
			ret = ret .. string.char(c)
		else
			return ret
		end
	end
	return ret
end


function fillCodeByZero(text, size)
    if #text>size then
        text = string.sub(txt,1,size)
    end
	while #text<size do
		text = "0"..text
	end
	return text
end

function convertUnitIdTo12bitOctal(unitId)
	local TN = unitId + 128
	local str = ""
	for i = 0, 3 do
		local octalDigit
		octalDigit = TN % 8
		TN = (TN - octalDigit) / 8
		str = octalDigit .. str
	end
	return str
end


function getIFFM2(tbl)
	--return convertUnitIdTo12bitOctal(tbl.unitId)
    return "auto"
end

function onChange_IFFM2(text)
	local txt =  custom_input_filter_octal( text )
	return #txt <= 4 and txt or string.sub(txt,1,4)
end

function onFocus_IFFM2(text, focused)
    if focused and (#text==0 or text=="auto") then
        return ""
    elseif not focused and #text==0 then
        return "auto"
	elseif not focused and #text > 0 then
		return fillCodeByZero(text, 4)
    else
        return text
	end
end

function getGearPositions(minMaxCoords, defaultCoordPercentage, defaultArgValue)
	local minCoord_y = minMaxCoords[1]
	local maxCoord_y = minMaxCoords[2]
	
	local strutTravel = minCoord_y - maxCoord_y
	
	local defaultCoord_y = minCoord_y - strutTravel * defaultCoordPercentage
	
	local strutPosMin = defaultCoord_y - minCoord_y
	local strutPosMax = defaultCoord_y - maxCoord_y
	local strutPosDefault = strutPosMax - defaultArgValue * strutTravel

	return defaultCoord_y, strutPosMin, strutPosMax, strutPosDefault
end

canopyArg					= 38

g_suit_effectiveness		= 1

noPassivCounterm = {
	CMDS_Edit 				= false,
	SingleChargeTotal 		= 0,
	chaff 					= {default = 0, increment = 0, chargeSz = 0},
	flare 					= {default = 0, increment = 0, chargeSz = 0}
}

passivCounterm = {
	CMDS_Edit 				= true,
	SingleChargeTotal 		= 252,
	chaff 					= {default = 126, increment = 18, chargeSz = 1},
	flare 					= {default = 56, increment = 8, chargeSz = 1}
}

-- WARNING. Do not add any refueling probe connectors to the 3D model. It gonna prevent from coordinates to be used.
-- KC-130 and KC-135MPRS have 'floating' drogue contact point lateral positioning.
-- The short probe coordinates were adjusted to account for this.
air_refuel_receptacle_pos_std = {5.66, 0.483, 0.311}
air_refuel_receptacle_pos_EE_EH = {5.818, 0.483, 0.311}
air_refuel_receptacle_pos_AD_AZ = {6.002, 0.429, 0.096}

panelRadio_CE_BE_EE = {
	[1] = {  
		name = _("TRAP-136"),
		range = {min = 118.0, max = 399.975},
		channels = {
			[1] = { name = _("Channel 1"), default = 118.0, modulation = _("AM"), connect = true},
			[2] = { name = _("Channel 2"), default = 119.25, modulation = _("AM")},
			[3] = { name = _("Channel 3"), default = 122.0, modulation = _("AM")},
			[4] = { name = _("Channel 4"), default = 126.5, modulation = _("AM")},
			[5] = { name = _("Channel 5"), default = 127.0, modulation = _("AM")},
			[6] = { name = _("Channel 6"), default = 129.0, modulation = _("AM")},
			[7] = { name = _("Channel 7"), default = 131.0, modulation = _("AM")},
			[8] = { name = _("Channel 8"), default = 133.0, modulation = _("AM")},
			[9] = { name = _("Channel 9"), default = 141.0, modulation = _("AM")},
			[10] = { name = _("Channel 10"), default = 250.5, modulation = _("AM")},
			[11] = { name = _("Channel 11"), default = 251.0, modulation = _("AM")},
			[12] = { name = _("Channel 12"), default = 253.0, modulation = _("AM")},
			[13] = { name = _("Channel 13"), default = 254.0, modulation = _("AM")},
			[14] = { name = _("Channel 14"), default = 257.0, modulation = _("AM")},
			[15] = { name = _("Channel 15"), default = 260.0, modulation = _("AM")},
			[16] = { name = _("Channel 16"), default = 261.0, modulation = _("AM")},
			[17] = { name = _("Channel 17"), default = 262.0, modulation = _("AM")},
			[18] = { name = _("Channel 18"), default = 263.0, modulation = _("AM")},
			[19] = { name = _("Channel 19"), default = 267.0, modulation = _("AM")},
			[20] = { name = _("Channel 20"), default = 270.0, modulation = _("AM")},
		}			
	},
	
	[2] = {  
		name = _("TRAP-137B"),
		range = {min = 225.0, max = 399.975},
		channels = {
			[1] = { name = _("Channel 1"), default = 225.0, modulation = _("AM")},
			[2] = { name = _("Channel 2"), default = 230.0, modulation = _("AM")},
			[3] = { name = _("Channel 3"), default = 240.0, modulation = _("AM")},
			[4] = { name = _("Channel 4"), default = 250.5, modulation = _("AM")},
			[5] = { name = _("Channel 5"), default = 251.0, modulation = _("AM")},
			[6] = { name = _("Channel 6"), default = 256.0, modulation = _("AM")},
			[7] = { name = _("Channel 7"), default = 257.0, modulation = _("AM")},
			[8] = { name = _("Channel 8"), default = 262.0, modulation = _("AM")},
			[9] = { name = _("Channel 9"), default = 263.0, modulation = _("AM")},
			[10] = { name = _("Channel 10"), default = 267.0, modulation = _("AM")},
			[11] = { name = _("Channel 11"), default = 270.0, modulation = _("AM")},
			[12] = { name = _("Channel 12"), default = 254.0, modulation = _("AM")},
			[13] = { name = _("Channel 13"), default = 264.0, modulation = _("AM")},
			[14] = { name = _("Channel 14"), default = 266.0, modulation = _("AM")},
			[15] = { name = _("Channel 15"), default = 265.0, modulation = _("AM")},
			[16] = { name = _("Channel 16"), default = 252.0, modulation = _("AM")},
			[17] = { name = _("Channel 17"), default = 268.0, modulation = _("AM")},
			[18] = { name = _("Channel 18"), default = 271.0, modulation = _("AM")},
			[19] = { name = _("Channel 19"), default = 275.0, modulation = _("AM")},
			[20] = { name = _("Channel 20 (N/A in 'Easy Comms')"), default = 360.0, modulation = _("AM")},
		}			
	}
}

function makeAttribute(WorldID, refuelable)
	local res = {wsType_Air, wsType_Airplane, wsType_Fighter, WorldID, "Multirole fighters"}
	
	if refuelable == true then
		res[#res + 1] = "Refuelable"
	end
	
	return res
end

local defaultLblAndCtrlGap			= 20
local optionLblWidth 				= 220
local optionCtrlPosX 				= optionLblWidth + defaultLblAndCtrlGap
local optionCtrlWidthLong 			= 130
local optionCtrlWidthShort 			= 75

function makeAircraftProperties(...)
	
	-- Properties common for all aircraft versions
	local props = {
		{ id = "RadarCoverSettings", control = 'comboList', label = _('Radar Cover State'), playerOnly = true, 
		  values = {{id = 1, dispName = _("By global option")},
					{id = 2, dispName = _("Force installed")},
					{id = 3, dispName = _("Force not installed")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthLong
		},
		
		{ id = "MissSimplLock", control = 'comboList', label = _('Simplified Missile Locking'), playerOnly = true, 
		  values = {{id = 1, dispName = _("No")},
					{id = 2, dispName = _("Yes")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},
	
		{ id = "ChaffMultiNumber", control = 'comboList', label = _('Chaff Burst Count'), playerOnly = true, 
		  values = {{id = 1, dispName = _("1")},
					{id = 2, dispName = _("2")},
					{id = 3, dispName = _("3")},
					{id = 4, dispName = _("4")},
					{id = 5, dispName = _("6")},
					{id = 6, dispName = _("8")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},
		
		{ id = "ChaffMultiTime", control = 'comboList', label = _('Chaff Burst Interval'), playerOnly = true, 
		  values = {{id = 1, dispName = _("0.1 s")},
					{id = 2, dispName = _("0.2 s")},
					{id = 3, dispName = _("0.3 s")},
					{id = 4, dispName = _("0.4 s")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},
		
		{ id = "ChaffProgramNumber", control = 'comboList', label = _('Chaff Salvo Count'), playerOnly = true, 
		  values = {{id = 1, dispName = _("1")},
					{id = 2, dispName = _("2")},
					{id = 3, dispName = _("4")},
					{id = 4, dispName = _("8")},
					{id = 5, dispName = _("Continuous")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},
		
		{ id = "ChaffProgramTime", control = 'comboList', label = _('Chaff Salvo Interval'), playerOnly = true, 
		  values = {{id = 1, dispName = _("1.0 s")},
					{id = 2, dispName = _("2.0 s")},
					{id = 3, dispName = _("3.0 s")},
					{id = 4, dispName = _("4.0 s")},
					{id = 5, dispName = _("5.0 s")},
					{id = 6, dispName = _("8.0 s")},
					{id = 7, dispName = _("Random")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},
		
		{ id = "FlareMultiNumber", control = 'comboList', label = _('Flare Burst Count'), playerOnly = true, 
		  values = {{id = 1, dispName = _("1")},
					{id = 2, dispName = _("2")},
					{id = 3, dispName = _("4")},
					{id = 4, dispName = _("8")},
					{id = 5, dispName = _("Continuous")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},
		
		{ id = "FlareMultiTime", control = 'comboList', label = _('Flare Burst Interval'), playerOnly = true, 
		  values = {{id = 1, dispName = _("3.0 s")},
					{id = 2, dispName = _("4.0 s")},
					{id = 3, dispName = _("6.0 s")},
					{id = 4, dispName = _("8.0 s")},
					{id = 5, dispName = _("10.0 s")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},
		
		{ id = "GunBurstSettings", control = 'comboList', label = _('Gun Burst Settings'), playerOnly = true, 
		  values = {{id = 0, dispName = _("Off")},
					{id = 1, dispName = _("Burst")},
					{id = 2, dispName = _("0.5 s")},
					{id = 3, dispName = _("1 s")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},	
		
		{ id = "RocketBurstF1", control = 'comboList', label = _('F1 Rocket Launcher Burst Count'), playerOnly = true, 
		  values = {{id = 1, dispName = _("6")},
					{id = 2, dispName = _("12")},
					{id = 3, dispName = _("18")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},
		
		{ id = "RocketBurstF4", control = 'comboList', label = _('F4/JL100 Rocket Launchers Burst Count'), playerOnly = true, 
		  values = {{id = 1, dispName = _("1")},
					{id = 2, dispName = _("3")},
					{id = 3, dispName = _("6")},
					{id = 4, dispName = _("18")}},
		  defValue  = 1,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl     = optionCtrlWidthShort
		},
		
		{ id = "IFFMode2Code", control = 'editbox', label = _('IFF Mode 2 Code'), getDefault = getIFFM2, onChange = onChange_IFFM2, onFocus = onFocus_IFFM2, playerOnly = true,
		  wLbl		= optionLblWidth,
		  xCtrl		= optionCtrlPosX,
		  wCtrl 	= optionCtrlWidthShort
		},
	}
	
	-- Add specific properties, added via the variable arguments list
	local arg = {...}
	for i, v in ipairs(arg) do
		props[#props + 1] = {}
		copyTable(v, props[#props])
	end
	
	return props
end

-- F1EE
extraAircraftProp_INSStartMode =
	{ id = "INSStartMode", control = 'comboList', label = _('INS start position always correct'), playerOnly = true, 
	  values = {{id = 1, dispName = _("Yes")},
				{id = 2, dispName = _("No")}},
	  defValue  = 1,
	  wLbl		= optionLblWidth,
	  xCtrl		= optionCtrlPosX,
	  wCtrl     = optionCtrlWidthShort
	}

-- F1EE
extraAircraftProp_RWR_type =
	{ id = "RWR_type", control = 'comboList', label = _('RWR type'), playerOnly = true, -- TODO: to enable for AI
	  values = {{id = "ALR_300", dispName = _("ALR-300"), value = 0.0},
				{id = "BF", dispName = _("BF"), value = 0.5}},
	  defValue  = "ALR_300",
	  wLbl		= optionLblWidth,
	  xCtrl		= optionCtrlPosX,
	  wCtrl     = optionCtrlWidthShort,
	  arg 		= 994
	}

-- Two-seaters
extraAircraftProp_SoloFlight =
	{id = "SoloFlight", control = 'checkbox', label = _('Solo Flight'),
	  defValue 	= false,
	  weight 	= -80,
	  arg 		= 472,
	  argTbl	 = {[true] = 0.9, [false] = 0},
	  wLbl		= optionLblWidth,
	  xCtrl		= optionCtrlPosX,
	  wCtrl 	= optionCtrlWidthShort
	}
	
extraAircraftProp_NetControlPriority =
	{ id = "NetCrewControlPriority" , control = 'comboList', label = _('Aircraft Control Priority'), playerOnly = true, 
	  values = {{id =  0, dispName = _("Pilot")},
				{id =  1, dispName = _("Instructor")},
				{id = -1, dispName = _("Ask Always")},
				{id = -2, dispName = _("Equally Responsible")}},
	  defValue  = 1,
	  wLbl		= optionLblWidth,
	  xCtrl		= optionCtrlPosX,
	  wCtrl     = optionCtrlWidthLong
	}
	
-- CE, EE, BE
extraAircraftProp_Mode4IFFDisabled =
	{ id = "IFFMode4Disabled", control = 'comboList', label = _('IFF Mode 4 Transponder Installed'), playerOnly = true, 
	  values = {{id = 1, dispName = _("By global option")},
				{id = 2, dispName = _("Force installed")},
				{id = 3, dispName = _("Force not installed")}},
	  defValue  = 1,
	  wLbl		= optionLblWidth,
	  xCtrl		= optionCtrlPosX,
	  wCtrl     = optionCtrlWidthLong
	}


function makeMechAnimations(...)
	
	-- mechanimations common for all aircraft versions
	local animations = {
        Door0 = {
            {Transition = {"Close", "Open"},  Sequence = {{C = {{"VelType", 3}, {"Arg", canopyArg, "to", 0.9, "in", 2.75}}}}, Flags = {"Reversible"}},
            {Transition = {"Open", "Close"},  Sequence = {{C = {{"VelType", 3}, {"Arg", canopyArg, "to", 0.0, "in", 2.75}}}}, Flags = {"Reversible", "StepsBackwards"}},
            {Transition = {"Open", "Bailout"},Sequence = {{C = {{"TearCanopy", 0},},},},},
            {Transition = {"Any", "Bailout"}, Sequence = {{C = {{"Origin", "x", 2.871, "y", 0.677, "z", 0.0}, {"Impulse", 1, "tertiary", 0.001}, {"Arg", 149, "set", 1.0}}}}},
			{Transition = {"Any", "TearOff"}, Sequence = {{C = {{"TearCanopy", 0},},},},},
        },
		HeadLights = {
            {Transition = {"Any", "Retract"}, Sequence = {{C = {{"Arg", 427, "to", 0.0, "speed", 1 / 3, "sign", -1.0},},},},},
            {Transition = {"Any", "Taxi"},    Sequence = {{C = {{"Arg", 427, "to", 1.0, "speed", 1 / 3, "sign", 1.0},},},},},
            {Transition = {"Any", "High"},    Sequence = {{C = {{"Arg", 427, "to", 0.86, "speed", 1 / 3, "sign", 1.0},},},},},
        },
	}
	
	-- Add specific animations, added via the variable arguments list
	local arg = {...}
	for i, v in ipairs(arg) do
		for kv, vv in pairs(v) do
			animations[kv] = {}
			copyTable(vv, animations[kv])
		end
	end
	
	return animations
end

function makeLights(landingLightConnector, isTwoSeater, hasRefuelLight)
	
	--local projectors_color = {255, 216, 160, 0.4}
	local projectors_color = {255, 194, 97, 0.4}
	
	local landingTaxiLight
	
	do
		local power_up_t = 1
		
		landingTaxiLight = 
		{
			{
				typename = "spotlight", connector = landingLightConnector, argument = 209,
				range = 1500, angle_max = math.rad(30), angle_min = math.rad(10), angle_change_rate = math.rad(8),
				power_up_t = power_up_t, movable = true,
				color = projectors_color,
			},
			{
				typename = "spotlight", connector = landingLightConnector,
				range = 40, angle_max = math.rad(165), angle_min = math.rad(145), angle_change_rate = math.rad(-52),
				power_up_t = power_up_t, movable = true,
				color = projectors_color,
			},
		}
	end
	
	local function makeCockpitLights(isRearSeat)
	
		local arg_UV
		local arg_front_dash_integral
		local arg_front_dash_flood
		local arg_side_panels_integral
		local arg_side_panels_flood
		local arg_misc_integral
		local arg_AoA_integral
		
		if isRearSeat ~= true then
			-- forward seat
			
			arg_UV						= 975
			arg_front_dash_integral		= 69
			arg_front_dash_flood		= 974
			arg_side_panels_integral	= 970
			arg_side_panels_flood		= 973
			arg_misc_integral			= 971
			arg_AoA_integral			= 972
		else
			-- rear seat
			
			arg_UV						= 980
			arg_front_dash_integral		= 976
			arg_front_dash_flood		= 981
			arg_side_panels_integral	= 977
			arg_side_panels_flood		= 982
			arg_misc_integral			= 978
			arg_AoA_integral			= 979
		end

		return {
			typename = "collection",
			lights = {
				[1] = { -- 0 -- UV lights
					typename = "collection",
					lights = {
						{
							typename = "argumentlight", argument = arg_UV,
						},
					},
				},
				[2] = { -- 1 -- front dash, integral lighting
					typename = "collection",
						lights = {
						{
							typename = "argumentlight", argument = arg_front_dash_integral,
						},
					},
				},
				[3] = { -- 2 -- front dash, flood lights
					typename = "collection",
					lights = {
						{
							typename = "argumentlight", argument = arg_front_dash_flood,
						},
					},
				},
				[4] = { -- 3 -- side panels, integral lighting
					typename = "collection",
					lights = {
						{
							typename = "argumentlight", argument = arg_side_panels_integral,
						},
					},
				},
				[5] = { -- 4 -- side panels, flood lights
					typename = "collection",
					lights = {
						{
							typename = "argumentlight", argument = arg_side_panels_flood,
						},
					},
				},
				[6] = { -- 5 -- miscelaneous, integral lighting
					typename = "collection",
					lights = {
						{
							typename = "argumentlight", argument = arg_misc_integral,
						},
					},
				},
				[7] = { -- 6 -- AoA indicator, integral lighting
					typename = "collection",
					lights = {
						{
							typename = "argumentlight", argument = arg_AoA_integral,
						},
					},
				},
			},
		}
	end
	
	local cockpitLights
	
	if isTwoSeater ~= true then
		cockpitLights = {
			typename = "collection",
			lights = {
				[1] = makeCockpitLights(false), -- 0 -- forward seat
			}
		}
	else
		cockpitLights = {
			typename = "collection",
			lights = {
				[1] = makeCockpitLights(false), -- 0 -- forward seat
				[2] = makeCockpitLights(true), -- 1 -- rear seat
			}
		}
	end
	
	local formLightDimmerExposure = {{928, 0, 1, 0, 1}}
	local navLightsDimmerExposure = {{929, 0, 1, 0, 1}}
		
	local nav_lights_power_up_t = 0.052
		
	local lights = {
			[WOLALIGHT_LANDING_LIGHTS] = {
				typename = "collection",
				lights = {
					{
						typename = "collection",
						lights = landingTaxiLight
					},
				},
			},
			[WOLALIGHT_TAXI_LIGHTS] = {
				typename = "collection",
				lights = {
					{
						typename = "collection",
						lights = landingTaxiLight
					},
				},
			},
			[WOLALIGHT_NAVLIGHTS] = {
				typename = "collection",
				lights ={
				{typename = "argumentlight",  argument  = 190, power_up_t = nav_lights_power_up_t, exposure = navLightsDimmerExposure}, -- Red - left wing
				{typename = "argumentlight",  argument  = 191, power_up_t = nav_lights_power_up_t, exposure = navLightsDimmerExposure}, -- Green - right wing
				{typename = "argnatostrobelight", argument = 83, period = 1.3333, flash_time = 0.3333, exposure = navLightsDimmerExposure}, -- Bottom Strobe
				{typename = "argumentlight", argument = 802, power_up_t = nav_lights_power_up_t, exposure = navLightsDimmerExposure}, -- Tail Strobe Orange
				-- Tail Strobe White is not described here (not used by AI)
				},
			},
			[WOLALIGHT_FORMATION_LIGHTS] = {
				typename = "collection",
				lights ={
				{typename = "argumentlight",  argument = 200, exposure = formLightDimmerExposure}, -- CT/CR
				{typename = "argumentlight",  argument  = 201, exposure = formLightDimmerExposure},
				},
			},	
			-- Police light
			[WOLALIGHT_PROJECTORS] = {
				typename = "collection",
				lights ={
				--{typename = "argumentlight",  argument  = 905},
				{
					typename = "spotlight", connector = "PILOT_LIGHT", argument = 905,
					range = 1500, angle_max = math.rad(32), angle_min = math.rad(11), angle_change_rate = math.rad(8),
					power_up_t = 1, movable = true,
					color = projectors_color,
				},
				},
			},
			[WOLALIGHT_CABIN_NIGHT] = cockpitLights,
	}
		
	if hasRefuelLight == true then
		lights[WOLALIGHT_REFUEL_LIGHTS] = {
			typename = "collection",
			lights = {
				{typename = "argumentlight", argument  = 927},
	}
		}
	
		local color = projectors_color
		local power_up_t = 1
	
		lights[WOLALIGHT_AUX_LIGHTS] = {
			typename = "collection",
			lights = {
	{
					typename = "spotlight", connector = "REFUEL_SPOT_PTR", argument = 903,
					range = 150, angle_max = math.rad(32), angle_min = math.rad(11), angle_change_rate = math.rad(8),
					power_up_t = power_up_t, cool_down_t = power_up_t, movable = true,
					color = color,
		},
			}
		}
	end
	
	return {typename = "collection", lights = lights}
end
	
function makeNetAnimation(isTwoSeater, hasRefuelLight)
	local res =
	{
		11, -- Right aileron
		12,	-- Left aileron
		13, -- Right slat
		14,	-- Left slat
		15, -- Right stabilizer
		16,	-- Left stabilizer
		18, -- Rudder
		23,	-- Wheel Chocks
		28, -- Afterburner
		35, -- Brake chute deployment position
		36,	-- Brake chute deflection up/down
		37,	-- Brake chute deflection left/right
		canopyArg, -- Canopy position
		75, -- GPU presence
		86, -- Nosecones position
		90,	-- Nozzle position
		115, -- Right gear door
		116, -- Left gear door
		117, -- Nose gear door
		120, -- Right spoiler
		123, -- Left spoiler
		126, -- Right inner flap
		127, -- Right outer flap
		128, -- Left inner flap
		129, -- Left outer flap
		149, -- Canopy embrittlement
		182, -- Right airbrake
		184, -- Left airbrake
		282, -- Additional air intake right up
		283, -- Additional air intake right low
		284, -- Additional air intake left up
		285, -- Additional air intake left low
		
		308, -- Wingtip pylon left
		309, -- Outboard pylon left
		310, -- Inboard pylon left
		311, -- Centerline pylon
		312, -- Inboard pylon right
		313, -- Outboard pylon right
		314, -- Wingtip pylon right
		
		338, -- Right wing flex
		339, -- Left wing flex
		340, -- Right wing tors
		341, -- Left wing tors
		427, -- Landing light pos
		705, -- Radar cover
		750, -- Countermeasure number flare left
		751, -- Countermeasure number chaff right
		752, -- Countermeasure type left
		753, -- Countermeasure type right
		754, -- Countermeasure number flare right
		755, -- Countermeasure number chaff left
		780, -- ADI roll
		781, -- ADI pitch
		782, -- ADI bearing
		803, -- Tail Strobe White (flashing, is used only by flyable)
		845, -- Access ladder
		905, -- Police light
		911, -- Canopy hanger
		
		-- Data
		920, -- Is human
		924, -- GPU state - off, start, idle, etc..
		925, -- Compressor stall state
		926, -- Engine starter
		928, -- Formation light brightness
		929, -- Navigation lights brightness
		
		-- TODO: to move to EE/M specific section
		
	}
	
	if isTwoSeater == true then
		-- Add two-seater specific net animation here
	end
	
	if hasRefuelLight == true then
		-- Air Refueling Light extension/retraction
		res[#res + 1] = 997
	end
	
	return res
end

F1_all_versions = {
	Rate 						= 40, -- RewardPoint in Multiplayer
	
	shape_table_data = 
	{
		{
			life  	 			= 18, -- lifebar
			vis   	 			= 3, -- visibility gain
			fire  	 			= {300, 6}, -- Fire on the ground after destoyed: 300sec 2m
			classname 			= "lLandPlane",
			positioning 		= "BYNORMAL",		
		},
		{
			fire  				= {300, 6},
			classname 			= "lLandPlane",
			positioning 		= "BYNORMAL",		
		},
	},
	
	mapclasskey 				= "P0091000024",
	Categories 					= {"{78EFB7A2-FD52-4b57-A6A6-3BF0E1D6555F}", "Interceptor",},
	
	H_max 					 	= 15240, -- m --50000 ft practical ceiling. 55000 ft absolute.
	average_fuel_consumption 	= 0.5, -- this is highly relative, but good estimates are 36-40l/min = 28-31kg/min = 0.47-0.52kg/s -- 45l/min = 35kg/min = 0.583kg/s
	CAS_min 					= 50, -- if this is not OVERAL FLIGHT TIME, but jus LOITER TIME, than it sholud be 10-15 minutes.....CAS capability in minute (for AI)
	V_opt 						= 300,-- Cruise speed (for AI)
	Mach_max 					= 2.1, -- Max speed in Mach (for AI)
	Ny_min 						= -3, -- Min G (for AI)
	Ny_max 						= 7.2,  -- Max G (for AI)
	Ny_max_e 					= 6.0,  -- Max G (for AI)
	bank_angle_max 				= 85, -- Max bank angle (for AI)
	
	has_afteburner 				= true, -- AFB yes/no
	has_speedbrake 				= true, -- Speedbrake yes/no
	has_differential_stabilizer = false,
	bigParkingRamp				= false,
	
	stores_number				= 9,
	brakeshute_name				= "Mirage-F1_parachute_1", -- Landing - brake chute visual shape after separation
	
	tanker_type 				= 0, -- Tanker type if the plane is airrefuel capable
	is_tanker 					= false, -- Tanker yes/no
	
	wing_area 					= 25, -- wing area in m2
	wing_span 					= 8.4, -- wing span in m
	wing_type 					= 0,
	length 						= 15.3, -- full lenght in m
	height 						= 4.5, -- height in m
	flaps_maneuver 				= 0.5, -- Max flaps in take-off and maneuver (0.5 = 1st stage; 1.0 = 2nd stage) (for AI)
	range 						= 3300, -- Max range in km (for AI)
	RCS 						= 5, -- Radar Cross Section m2
	IR_emission_coeff 			= 0.8, -- Normal engine -- IR_emission_coeff = 1 is Su-27 without afterburner. It is reference.
	IR_emission_coeff_ab 		= 3, -- With afterburner
	wing_tip_pos 				= {-3.5, -0.04, 4.22}, -- wingtip coords for turbulence and AI taxi/collision logic

	engines_count				= 1, -- Engines count
	thrust_sum_max 				= 4867, -- thrust in kg (47.7kN)
	thrust_sum_ab 				= 6969, -- thrust in kg (68.3kN)
	
	----------------- LANDING GEAR
	nose_gear_wheel_diameter 	= 0.346, -- in m
	main_gear_wheel_diameter 	= 0.608, -- in m

	wheel_steering_angle_max	= math.rad(45),
		
	engines_nozzles = {
		[1] = 
		{
			pos 				=  {-8.0,-0.111,0.0}, 	-- nozzle coords
			elevation   		=  0, 					-- AFB cone elevation
			diameter	 		= 0.9,					-- AFB cone diameter
			exhaust_length_ab   = 9, 					-- lenght in m
			exhaust_length_ab_K = 0.5, 					-- AB animation
			smokiness_level 	= 0.2,				
		}, -- end of [1]
	}, -- end of engines_nozzles
	
	fires_pos = {
		[1] = 	{-0.855,	0.593,	   0.0},	-- Fuselage					
		[2] = 	{-2.314,	0.178,	 2.065},	-- Wing inner Right
		[3] = 	{-2.332,	0.187,  -2.02},	    -- Wing inner Left
		[4] = 	{-2.836,	0.122,   2.775},	-- Wing middle Right
		[5] = 	{-2.973,	0.095,  -3.034},	-- Wing middle Left
		[6] = 	{-3.615,   -0.012,	 4.086},	-- Wing outer Right
		[7] = 	{-3.553,   -0.017,	-4.077},	-- Wing outer Left			
		[8] = 	{-5.249,     -0.1,	   0.0},	-- Engine
	}, -- end of fires_pos
	
	effects_presets = {
		{effect = "OVERWING_VAPOR", file = current_mod_path.."/Effects/MirageF1_overwingVapor.lua"},
	},
	
	-- No passive countermeasures by default
	passivCounterm 				= noPassivCounterm,
	-- Coordinates for passive countermeasures dispencers (when those are present)
	chaff_flare_dispenser = {
		[1] = 
		{
			dir 				= {0, -1, 0},
			pos 				= {-1.0, -0.82, -1.262},
		}, -- end of [1]
		[2] = 
		{
			dir 				= {0, -1, 0},
			pos 				= {-1.0, -0.82, 1.262},
		}, -- end of [2]
	}, -- end of chaff_flare_dispenser

	-- Sensors
	detection_range_max		 	= 70,
	radar_can_see_ground 	 	= true,
	
	Sensors = {
		RADAR 					= "RDY",
		IRST 					= nil,
		RWR 					= "Abstract RWR"
	},
	
	CanopyGeometry = {
		azimuth   				= {-160.0, 160.0}, -- pilot view horizontal (AI)
		elevation 				= {-50.0, 90.0} -- pilot view vertical (AI)
	},		
	
	HumanRadio = {		
		frequency 				= 127.5,  -- Radio Freq
		editable 				= true,
		minFrequency 			= 100.000,
		maxFrequency 			= 400.000,
		modulation 				= MODULATION_AM
	},
		
	SFM_Data = {
		engine = 
		{
			type	=	"TurboJet",
			Nmg		=	67.5,
			MinRUD	=	0,
			MaxRUD	=	1,
			MaksRUD	=	0.85,
			ForsRUD	=	0.91,
			typeng	=	1,
			hMaxEng	=	19.5,
			dcx_eng	=	0.0085,  -- Affects drag of engine when shutdown
			                     -- cemax/cefor affect sponginess of elevator/inertia at slow speed
			                     -- affects available g load apparently
			cemax	=	1.24,
			cefor	=	2.56,
			-- following affect thrust/max alt
			dpdh_m	=	3270,  -- Loss of thrust in N per 1000 m
			dpdh_f	=	6600, -- Loss of thrust in N per 1000 m
			table_data = 
			{
			         --   M		 Pmax        Pfor
				[1] = 	{0,	    49000,	    70600},
				[2] = 	{0.2,	45700,	    67700}, -- 132 kts at SL
				[3] = 	{0.3,	44900,	    68700}, -- 198 kts at SL
				[4] = 	{0.4,	44800,	    69700}, -- 265 kts at SL
				[5] = 	{0.6,	46000,	    74600}, -- 397 kts at SL
				[6] = 	{0.7,	47000,	    78500},
				[7] = 	{0.8,	48600,	    82400},
				[8] = 	{0.9,	50000,	    87500},
				[9] = 	{1,	    51500,	    93200},
				[10] = 	{1.1,	53000,	    98100},
				[11] = 	{1.2,	54500,	    104000},
				[12] = 	{1.3,	56500,	    109900},
				[13] = 	{1.5,	59000,	    121600},
				[14] = 	{1.8,	70000,	    144000},
				[15] = 	{2,	    70000,	    155800},
				[16] = 	{2.2,	90000,	    167530},
				[17] = 	{2.5,	47700,	    162000},
				[18] = 	{3,	    20000,	    40000},
			}, -- end of table_data
			-- M - Mach number
			-- Pmax - Engine thrust at military power
			-- Pfor - Engine thrust at AFB
		}, -- end of engine
	},
---------------------------------------------------------------------------------------------------------------------------------------------	
	Failures = 
	{
		{ id = "battery_fail", label = _("Battery failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "alt1_fail", label = _("Alternator 1 failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "alt2_fail", label = _("Alternator 2 failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "tr1_fail", label = _("Transformer 1 failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "tr2_fail", label = _("Transformer 2 failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "triphase_inv_fail", label = _("Three-phase inverter failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "miss_bus_fail", label = _("Missile bus failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		
		{ id = "left_fuel_pump_fail", label = _("Left fuel pump failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "right_fuel_pump_fail", label = _("Right fuel pump failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "detotalizer_fail", label = _("Fuel detotalizer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "fuel_gauges_fail", label = _("Fuel gauges failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "fuel_intercom_fail", label = _("Fuel crossfeed valve failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		--{ id = "fuel_pressurization_fail", label = _("Fuel pressurization failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "left_wing_transfer_fail", label = _("Left wing fuel tank transfer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "right_wing_transfer_fail", label = _("Right wing fuel tank transfer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "front_central_transfer_fail", label = _("Front central fuel tank transfer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "left_front_transfer_fail", label = _("Left front fuel tank transfer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "right_front_transfer_fail", label = _("Right front fuel tank transfer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "left_rear_transfer_fail", label = _("Left rear fuel tank transfer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "right_rear_transfer_fail", label = _("Right rear fuel tank transfer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "external_tanks_transfer_fail", label = _("External fuel tanks transfer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "left_wing_leaks", label = _("Left wing fuel tank leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "right_wing_leaks", label = _("Right wing fuel tank leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "front_central_leaks", label = _("Front central fuel tank leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "left_front_leaks", label = _("Left front fuel tank leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "right_front_leaks", label = _("Right front fuel tank leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "left_rear_leaks", label = _("Left rear fuel tank leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "right_rear_leaks", label = _("Right rear fuel tank leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "left_feeder_leaks", label = _("Left feeder fuel tank leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "right_feeder_leaks", label = _("Right feeder fuel tank leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "fuel_accu_leaks", label = _("Fuel accumulator leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		
		{ id = "left_airbrake_fail", label = _("Left airbrake failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "right_airbrake_fail", label = _("Right airbrake failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gear_lever_fail", label = _("Gear actuation failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gear_down_lock_fail", label = _("Gear locking failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gear_nose_stuck", label = _("Nose gear stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gear_left_stuck", label = _("Left gear stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gear_right_stuck", label = _("Right gear stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "brakes_fail", label = _("Wheel brakes failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "chute_fail", label = _("Drag chute failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "electropump_fail", label = _("Electropump failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "hydr1_leaks", label = _("Hydraulic circuit 1 leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "hydr1_reserv_leaks", label = _("Hydraulic reservoir 1 leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "hydr1_pump_fail", label = _("Hydraulic pump 1 failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "hydr2_leaks", label = _("Hydraulic circuit 2 leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "hydr2_reserv_leaks", label = _("Hydraulic reservoir 2 leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "hydr2_pump_fail", label = _("Hydraulic pump 2 failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "hydr_serv_leaks", label = _("Hydraulic servitudes circuit leaks"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "flap_left_stuck", label = _("Left flap stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "flap_right_stuck", label = _("Right flap stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "flaps_stuck", label = _("Both flaps stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "slat_inner_left_stuck", label = _("Left inner slat stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "slat_inner_right_stuck", label = _("Right inner slat stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "slat_outer_left_stuck", label = _("Left outer slat stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "slat_outer_right_stuck", label = _("Right outer slat stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "slats_stuck", label = _("All slats stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		
		{ id = "pitot_heat_fail", label = _("Pitot tube heating failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "altitude_chain_fail", label = _("Altitude chain failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "mach_chain_fail", label = _("Mach chain failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "incidometer_fail", label = _("Incidometer failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "incidometer_blockage_fail", label = _("Incidometer stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "anemo_central_fail", label = _("Anemometry central failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		
		{ id = "trim_pitch_fail", label = _("Pitch trim fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "trim_roll_fail", label = _("Roll trim fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "trim_yaw_fail", label = _("Yaw trim fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "trim_elect_supply", label = _("Trim electric failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "yaw_damper_fail", label = _("Yaw damper failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "pilot_aids_1_fail", label = _("Pilot aids 1 electric supply failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "pilot_aids_2_fail", label = _("Pilot aids 2 electric supply failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "pitch_chain_fail", label = _("Pitch control chain failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "ap_global_fail", label = _("Autopilot failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		
		{ id = "broken_guards", label = _("Engine nozzle vanes stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "oil_fail", label = _("Engine oil pump failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "nosecone_stuck", label = _("Engine nosecones stuck"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "nosecone_stuck_forward", label = _("Engine nosecones stuck forwards"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "nosecone_stuck_backward", label = _("Engine nosecones stuck backwards"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "start_fail", label = _("Engine starter failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "ignition_fail", label = _("Engine ignition failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "total_comp_stall", label = _("Total compressor stall"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "partial_comp_stall", label = _("Partial compressor stall"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "overspeed_fail", label = _("Engine overspeed failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "discharge_valves_fail", label = _("Engine discharge valves failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		--{ id = "accessory_support_fail", label = _("Engine accessory supports failure"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "engine_fire", label = _("Engine fire"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "AB_fire", label = _("Afterburner fire"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "compressor_damage", label = _("Engine compressor damaged"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "engine_flameout", label = _("Engine flameout"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		
		{ id = "cabin_temp_fail", label = _("Cabin temperature regulation fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "equip_temp_fail", label = _("Equipment bay temperature regulation fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "oxygen_regulator_fail", label = _("Oxygen regulator fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		
		{ id = "gyros_general_BSM_fail", label = _("Gyroscopic central fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gyros_main_att_fail", label = _("Main attitude gyroscope fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gyros_main_fail", label = _("Main directional gyroscope fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gyros_emergency_fail", label = _("Emergency directional gyroscope fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gyros_att_indicator", label = _("Spherical indicator fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		{ id = "gyros_temp_drift", label = _("Excessive gyro drift until realignment"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		
		{ id = "BARAX_fail", label = _("BARAX emission fail"),	enable = false, hh = 0, mm = 0, mmint = 1, prob = 100 },
		
	},
		
	AddPropAircraft = makeAircraftProperties(),
}
